<?php

namespace Drupal\Tests\commerce_promotion\Kernel\Plugin\Commerce\PromotionOffer;

use Drupal\Tests\commerce_order\Kernel\OrderKernelTestBase;
use Drupal\commerce_order\Entity\Order;
use Drupal\commerce_order\Entity\OrderItem;
use Drupal\commerce_price\Price;
use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\commerce_promotion\Entity\Promotion;

/**
 * Tests the percentage off offer for order items.
 *
 * @coversDefaultClass \Drupal\commerce_promotion\Plugin\Commerce\PromotionOffer\OrderItemPercentageOff
 *
 * @group commerce
 */
class OrderItemPercentageOffTest extends OrderKernelTestBase {

  /**
   * The test order.
   *
   * @var \Drupal\commerce_order\Entity\OrderInterface
   */
  protected $order;

  /**
   * The test variation.
   *
   * @var \Drupal\commerce_product\Entity\ProductVariationInterface
   */
  protected $variation;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'commerce_promotion',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('commerce_promotion');
    $this->installConfig(['commerce_promotion']);
    $this->installSchema('commerce_promotion', ['commerce_promotion_usage']);

    $variation = ProductVariation::create([
      'type' => 'default',
      'sku' => strtolower($this->randomMachineName()),
      'price' => [
        'number' => '9.99',
        'currency_code' => 'USD',
      ],
    ]);
    $variation->save();
    $this->variation = $variation;

    $product = Product::create([
      'type' => 'default',
      'title' => 'My product',
      'variations' => [$variation],
    ]);
    $product->save();

    $this->order = Order::create([
      'type' => 'default',
      'state' => 'completed',
      'mail' => 'test@example.com',
      'ip_address' => '127.0.0.1',
      'order_number' => '6',
      'uid' => $this->createUser(),
      'store_id' => $this->store,
      'order_items' => [],
    ]);
  }

  /**
   * Tests the display-inclusive offer.
   *
   * @covers ::apply
   */
  public function testDisplayInclusive() {
    $order_item = OrderItem::create([
      'type' => 'default',
      'quantity' => '2',
      'unit_price' => $this->variation->getPrice(),
      'purchased_entity' => $this->variation->id(),
    ]);
    $order_item->save();
    $this->order->addItem($order_item);
    $this->order->save();

    // Starts now, enabled. No end time.
    $promotion = Promotion::create([
      'name' => 'Promotion 1',
      'order_types' => [$this->order->bundle()],
      'stores' => [$this->store->id()],
      'status' => TRUE,
      'offer' => [
        'target_plugin_id' => 'order_item_percentage_off',
        'target_plugin_configuration' => [
          'display_inclusive' => TRUE,
          'percentage' => '0.50',
        ],
      ],
      'weight' => -1,
    ]);
    $promotion->save();

    $this->container->get('commerce_order.order_refresh')->refresh($this->order);
    $this->order = $this->reloadEntity($this->order);
    /** @var \Drupal\commerce_order\Entity\OrderItemInterface $order_item */
    $order_item = $this->reloadEntity($order_item);

    $this->assertEquals(new Price('4.99', 'USD'), $order_item->getUnitPrice());
    $this->assertEquals(new Price('9.98', 'USD'), $order_item->getTotalPrice());
    $this->assertEquals(new Price('9.98', 'USD'), $order_item->getAdjustedTotalPrice());
    $this->assertEquals(1, count($order_item->getAdjustments()));
    $adjustment = $order_item->getAdjustments()[0];
    $this->assertEquals('Discount', $adjustment->getLabel());
    $this->assertEquals(new Price('-10.00', 'USD'), $adjustment->getAmount());
    $this->assertEquals('0.50', $adjustment->getPercentage());
    $this->assertTrue($adjustment->isIncluded());
    $this->order->recalculateTotalPrice();
    $this->assertEquals(new Price('9.98', 'USD'), $this->order->getTotalPrice());

    $another_promotion = Promotion::create([
      'name' => 'Promotion 2',
      'order_types' => [$this->order->bundle()],
      'stores' => [$this->store->id()],
      'status' => TRUE,
      'offer' => [
        'target_plugin_id' => 'order_item_percentage_off',
        'target_plugin_configuration' => [
          'display_inclusive' => TRUE,
          'percentage' => '0.5',
        ],
      ],
      'weight' => 0,
    ]);
    $another_promotion->save();
    $this->container->get('commerce_order.order_refresh')->refresh($this->order);
    $this->order = $this->reloadEntity($this->order);
    /** @var \Drupal\commerce_order\Entity\OrderItemInterface $order_item */
    $order_item = $this->reloadEntity($order_item);
    $this->assertEquals(new Price('2.49', 'USD'), $order_item->getUnitPrice());
    $this->assertEquals(new Price('4.98', 'USD'), $order_item->getTotalPrice());
    $this->assertEquals(new Price('4.98', 'USD'), $order_item->getAdjustedTotalPrice());
    $adjustments = $order_item->getAdjustments();
    $this->assertEquals(2, count($adjustments));
    $this->assertEquals('Discount', $adjustments[0]->getLabel());
    $this->assertEquals(new Price('-10', 'USD'), $adjustments[0]->getAmount());
    $this->assertEquals(new Price('-5', 'USD'), $adjustments[1]->getAmount());
    $this->assertEquals('0.5', $adjustments[0]->getPercentage());
    $this->assertEquals('0.5', $adjustments[1]->getPercentage());
    $this->assertTrue($adjustments[0]->isIncluded());
    $this->assertTrue($adjustments[1]->isIncluded());
    $this->order->recalculateTotalPrice();
    $this->assertEquals(new Price('4.98', 'USD'), $this->order->getTotalPrice());

    $another_promotion = Promotion::create([
      'name' => 'Promotion 3',
      'order_types' => [$this->order->bundle()],
      'stores' => [$this->store->id()],
      'status' => TRUE,
      'offer' => [
        'target_plugin_id' => 'order_item_percentage_off',
        'target_plugin_configuration' => [
          'display_inclusive' => TRUE,
          'percentage' => '1',
        ],
      ],
      'weight' => 1,
    ]);
    $another_promotion->save();
    $this->container->get('commerce_order.order_refresh')->refresh($this->order);
    $this->order = $this->reloadEntity($this->order);
    /** @var \Drupal\commerce_order\Entity\OrderItemInterface $order_item */
    $order_item = $this->reloadEntity($order_item);
    $this->assertEquals(new Price('0', 'USD'), $order_item->getUnitPrice());
    $this->assertEquals(new Price('0', 'USD'), $order_item->getTotalPrice());
    $this->assertEquals(new Price('0', 'USD'), $order_item->getAdjustedTotalPrice());
    $adjustments = $order_item->getAdjustments();
    $this->assertEquals(3, count($adjustments));
    $this->assertEquals('Discount', $adjustments[0]->getLabel());
    $this->assertEquals(new Price('-10', 'USD'), $adjustments[0]->getAmount());
    $this->assertEquals(new Price('-5', 'USD'), $adjustments[1]->getAmount());
    $this->assertEquals(new Price('-4.98', 'USD'), $adjustments[2]->getAmount());
    $this->order->recalculateTotalPrice();
    $this->assertEquals(new Price('0', 'USD'), $this->order->getTotalPrice());
  }

  /**
   * Tests the non-display-inclusive offer.
   *
   * @covers ::apply
   */
  public function testNonDisplayInclusive() {
    $order_item = OrderItem::create([
      'type' => 'default',
      'quantity' => '2',
      'unit_price' => $this->variation->getPrice(),
      'purchased_entity' => $this->variation->id(),
    ]);
    $order_item->save();
    $this->order->addItem($order_item);
    $this->order->save();

    // Starts now, enabled. No end time.
    $promotion = Promotion::create([
      'name' => 'Promotion 1',
      'display_name' => '50% off',
      'order_types' => [$this->order->bundle()],
      'stores' => [$this->store->id()],
      'status' => TRUE,
      'offer' => [
        'target_plugin_id' => 'order_item_percentage_off',
        'target_plugin_configuration' => [
          'display_inclusive' => FALSE,
          'percentage' => '0.50',
        ],
      ],
    ]);
    $promotion->save();

    $this->container->get('commerce_order.order_refresh')->refresh($this->order);
    $this->order = $this->reloadEntity($this->order);
    /** @var \Drupal\commerce_order\Entity\OrderItemInterface $order_item */
    $order_item = $this->reloadEntity($order_item);

    $this->assertEquals(new Price('9.99', 'USD'), $order_item->getUnitPrice());
    $this->assertEquals(new Price('19.98', 'USD'), $order_item->getTotalPrice());
    $this->assertEquals(new Price('9.99', 'USD'), $order_item->getAdjustedTotalPrice());
    $this->assertEquals(1, count($order_item->getAdjustments()));
    $adjustment = $order_item->getAdjustments()[0];
    $this->assertEquals('50% off', $adjustment->getLabel());
    $this->assertEquals(new Price('-9.99', 'USD'), $adjustment->getAmount());
    $this->assertEquals('0.50', $adjustment->getPercentage());
    $this->assertFalse($adjustment->isIncluded());
    $this->order->recalculateTotalPrice();
    $this->assertEquals(new Price('9.99', 'USD'), $this->order->getTotalPrice());

    // Test with multiple promotions.
    $another_promotion = Promotion::create([
      'name' => 'Promotion 2',
      'display_name' => '100% off',
      'order_types' => [$this->order->bundle()],
      'stores' => [$this->store->id()],
      'status' => TRUE,
      'offer' => [
        'target_plugin_id' => 'order_item_percentage_off',
        'target_plugin_configuration' => [
          'display_inclusive' => FALSE,
          'percentage' => '1',
        ],
      ],
      'weight' => 10,
    ]);
    $another_promotion->save();
    $this->container->get('commerce_order.order_refresh')->refresh($this->order);
    $this->order = $this->reloadEntity($this->order);
    /** @var \Drupal\commerce_order\Entity\OrderItemInterface $order_item */
    $order_item = $this->reloadEntity($order_item);

    $this->assertEquals(new Price('9.99', 'USD'), $order_item->getUnitPrice());
    $this->assertEquals(new Price('19.98', 'USD'), $order_item->getTotalPrice());
    $this->assertEquals(new Price('0', 'USD'), $order_item->getAdjustedTotalPrice());
    $adjustments = $order_item->getAdjustments();
    $this->assertEquals(2, count($adjustments));
    $this->assertEquals('50% off', $adjustments[0]->getLabel());
    $this->assertEquals(new Price('-9.99', 'USD'), $adjustments[0]->getAmount());
    $this->assertEquals(new Price('-9.99', 'USD'), $adjustments[1]->getAmount());
    $this->assertEquals('0.50', $adjustments[0]->getPercentage());
    $this->assertEquals('1', $adjustments[1]->getPercentage());
    $this->assertFalse($adjustments[0]->isIncluded());
    $this->assertFalse($adjustments[1]->isIncluded());
    $this->order->recalculateTotalPrice();
    $this->assertEquals(new Price('0', 'USD'), $this->order->getTotalPrice());
  }

  /**
   * Tests applying both an inclusive and a non inclusive promotion.
   *
   * @covers ::apply
   */
  public function testStackingPromotions() {
    /** @var \Drupal\commerce_promotion\Entity\PromotionInterface $promotion */
    $promotion = Promotion::create([
      'name' => 'Promotion',
      'display_name' => '100% off',
      'order_types' => [$this->order->bundle()],
      'stores' => [$this->store->id()],
      'status' => TRUE,
      'offer' => [
        'target_plugin_id' => 'order_item_percentage_off',
        'target_plugin_configuration' => [
          'display_inclusive' => FALSE,
          'percentage' => '1',
        ],
      ],
      'weight' => 0,
    ]);
    $promotion->save();

    $inclusive_promotion = $promotion->createDuplicate();
    $inclusive_promotion->setDisplayName('Inclusive 100% off');
    $offer = $inclusive_promotion->getOffer();
    $offer->setConfiguration([
      'display_inclusive' => TRUE,
      'percentage' => '1',
    ]);
    $inclusive_promotion->setWeight(1);
    $inclusive_promotion->setOffer($offer);
    $inclusive_promotion->save();
    $order_item = OrderItem::create([
      'type' => 'default',
      'quantity' => '2',
      'unit_price' => $this->variation->getPrice(),
      'purchased_entity' => $this->variation->id(),
    ]);
    $order_item->save();
    $this->order->addItem($order_item);
    $this->order->save();
    $this->container->get('commerce_order.order_refresh')->refresh($this->order);
    $order_item = $this->reloadEntity($order_item);
    $adjustments = $order_item->getAdjustments();
    // The adjusted total is already reduced to 0, the inclusive promotion
    // shouldn't add an adjustment.
    $this->assertEquals(1, count($adjustments));
    $this->assertEquals(new Price('19.98', 'USD'), $order_item->getTotalPrice());
    $this->assertEquals(new Price('0', 'USD'), $order_item->getAdjustedTotalPrice());
    $this->assertEquals(new Price('9.99', 'USD'), $order_item->getUnitPrice());
    $this->assertEquals('100% off', $adjustments[0]->getLabel());
    $this->order->recalculateTotalPrice();
    $this->assertEquals(new Price('0', 'USD'), $this->order->getTotalPrice());
  }

}
