<?php

declare(strict_types=1);

namespace Drupal\commerce_kickstart_demo_assets\EventSubscriber;

use Drupal\Core\DefaultContent\PreImportEvent;
use Drupal\Core\DestructableInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Recipe\RecipeAppliedEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribes to the default content import events and handles post-import actions.
 *
 * This class listens to the default content import process. It uses a flag
 * to determine whether an import has taken place and, if so, triggers re-indexing
 * of the specified search index once the process is complete.
 *
 * Using `destruct` ensures that content is imported successfully before
 * performing any additional operations, such as re-indexing the specified
 * search index.
 */
final class DefaultContentSubscriber implements EventSubscriberInterface, DestructableInterface {

  /**
   * Flag which determines whether to index the DB.
   *
   * @var bool
   */
  private $import = FALSE;

  /**
   * Constructs a DefaultContentSubscriber object.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function onDefaultContentImport(PreImportEvent $event): void {
    $this->import = TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function destruct() {
    if ($this->import) {
      $index_storage = $this->entityTypeManager->getStorage('search_api_index');
      /** @var \Drupal\search_api\IndexInterface $index */
      $index = $index_storage->load('products');
      $index->indexItems();
      // Avoid running it multiple times.
      $this->import = FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      PreImportEvent::class => ['onDefaultContentImport'],
      // In case recipe is applied outside of Kernel scope having the following
      // event makes sure destruction is correctly triggered.
      RecipeAppliedEvent::class => ['destruct'],
    ];
  }

}
